//
// taputils.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

#include <stdlib.h>
#include <string.h>

#include "status.h"
#include "tap.h"
#include "tapecart.h"

#include "taputils.h"

// Enthusi's turbo tape loader
// https://codebase.c64.org/doku.php?id=base%3Aturbotape_loader_source
u8 turbo64Loader[171] = {
#include "c64/turbo_loader.h"
};
// basic stub to call the turbo loader
u8 turbo64BasicStub[] = {
#include "c64/basic_stub.h"
};

// tapecart loader by Ingo Korb
// https://github.com/ikorb/tapecart
//u8 tcrtLoader[171] = {
//#include "c64/tcrt_loader.h"
//};
//u8 tcrtVector[] = {
//0x02, 0x03, 0x51, 0x03
//};
//u8 tcrtBasicStarter[] = {
//  0x20, 0x60, 0xa6, // jsr $a660
//  0x20, 0x8e, 0xa6, // jsr $a68e
//  0x4c, 0xae, 0xa7, // jmp $a7ae
//  0x00              // $0800 must be zero
//};

// sid player
//u8 sidPlayer[] = {
//#include "c64/sid_player.h"
//};


TapUtils::TapUtils()
{
}

TapUtils::~TapUtils (void) { }

unsigned TapUtils::prg2Tap (tapConversions type, const u8 *prgBuf, unsigned prgLen,
		u8 *tapBuf, unsigned tapPos, const char *name, u8 *idxBuf)
{
	if(tapPos==0)
	{
		strcpy((char *)tapBuf+TAP_HDR, "C64-TAPE-RAW");
		tapBuf[TAP_VER]=1;
		for(int i=TAP_PLT; i<TAP_DAT; i++) tapBuf[i]=0;
		tapPos=TAP_DAT;
	}

	switch (type)
	{
		case KERNAL:
			appendIdx(idxBuf, tapPos, name);
			tapPos+=prg2Kernal(prgBuf, prgLen, NULL, tapBuf+tapPos, name);
			tapPos+=tapSpace(200, tapBuf+tapPos);
			break;
		case TURBOTAPE64:
			appendIdx(idxBuf, tapPos, name);
			tapPos+=prg2TurboTape64(prgBuf, prgLen, tapBuf+tapPos, name);
			tapPos+=tapSpace(200, tapBuf+tapPos);
			break;
		case TURBOTAPE64_WITH_LOADER:
			// enthusi's loader goes into the kernel header at $0351
			// basic stub calls sys $0351
			appendIdx(idxBuf, tapPos, "Turbo loader");
			tapPos+=prg2Kernal(turbo64BasicStub, sizeof(turbo64BasicStub), turbo64Loader, tapBuf+tapPos, name);
			tapPos+=tapSpace(150, tapBuf+tapPos);

			appendIdx(idxBuf, tapPos, name);
			tapPos+=prg2TurboTape64(prgBuf, prgLen, tapBuf+tapPos, name);
			tapPos+=tapSpace(500, tapBuf+tapPos);
			break;
		case TCRTLOADER:
			appendIdx(idxBuf, tapPos, name);
			if(prgBuf==NULL) {
				tapPos+=prg2Kernal(tcrtVector, sizeof(tcrtVector), tcrtLoader, tapBuf+tapPos, name);
			} else {
				tapPos+=prg2Kernal(tcrtVector, sizeof(tcrtVector), prgBuf, tapBuf+tapPos, name);
			}
			tapPos+=tapSpace(150, tapBuf+tapPos);
			break;
		default:
			break;
	}

	unsigned tap_size=tapPos-TAP_DAT;
	tapBuf[TAP_SIZ]=tap_size&0xff;
	tapBuf[TAP_SIZ+1]=(tap_size&0xff00)>>8;
	tapBuf[TAP_SIZ+2]=(tap_size&0xff0000)>>16;
	tapBuf[TAP_SIZ+3]=(tap_size&0xff000000)>>24;

	return tapPos;
}

unsigned TapUtils::p002Tap (tapConversions type, const u8 *p00Buf, unsigned p00Len,
		u8 *tapBuf, unsigned tapPos, const char *name, u8 *idxBuf)
{
	const char *p00name=(const char *)p00Buf+8;
	const u8 *prgBuf=p00Buf+26;
	unsigned prgLen=p00Len-26;

	if(type==TCRT)
	{
		return prg2Tapecart(prgBuf, prgLen, tapBuf, TCRT_MAX_SIZE, name);
	} else {
		return prg2Tap (type, prgBuf, prgLen, tapBuf, tapPos, p00name, idxBuf);
	}
}

//unsigned TapUtils::sid2Tap (tapConversions type, u8 *sidBuf, unsigned sidBufLen, unsigned sidLen,
//			u8 *tapBuf, unsigned tapPos, const char *name, u8 *idxBuf)
//{
	// prg: sidPlayer, sid data
//	if(sizeof(sidPlayer)+sidLen>sidBufLen) return -1;
//	for(int i=sidLen-1; i>=0; i--) sidBuf[sizeof(sidPlayer)+i]=sidBuf[i];
//	for(unsigned i=0; i<sizeof(sidPlayer); i++) sidBuf[i]=sidPlayer[i];

	//  patch last 2 bytes of sidPlayer with songlen
	//   songlen=sidLen-0x7c-2
//	unsigned sidDataLen=sidLen-0x7c-2;
//	sidBuf[sizeof(sidPlayer)-2]=sidDataLen&0xff;
//	sidBuf[sizeof(sidPlayer)-1]=(sidDataLen&0xff00)>>8;

//	return prg2Tap (type, sidBuf, sizeof(sidPlayer)+sidLen, tapBuf, tapPos, name, idxBuf);
//}

unsigned TapUtils::t642Tap (tapConversions type, const u8 *t64Buf, unsigned t64Len,
		u8 *tapBuf, unsigned tapPos, const char *name, u8 *idxBuf)
{
	unsigned numEntries=(t64Buf[0x25]<<8)|t64Buf[0x24];
	for (unsigned i=0; i<numEntries; i++)
	{
		char prgName[17];
		strncpy(prgName, (const char *)t64Buf+0x40+(i*0x20)+0x10, 16);
		prgName[16]='\0';

		unsigned startAddrLo=(unsigned)t64Buf[0x40+(i*0x20)+0x2];
		unsigned startAddrHi=(unsigned)t64Buf[0x40+(i*0x20)+0x3];
		unsigned endAddrLo=(unsigned)t64Buf[0x40+(i*0x20)+0x4];
		unsigned endAddrHi=(unsigned)t64Buf[0x40+(i*0x20)+0x5];
		unsigned startAddress=(startAddrHi<<8)|startAddrLo;
		unsigned endAddress=  (endAddrHi<<8)|endAddrLo;
		unsigned prgLen=(endAddress-startAddress)+2;

		unsigned offset1=(unsigned)t64Buf[0x40+(i*0x20)+0x8];
		unsigned offset2=(unsigned)t64Buf[0x40+(i*0x20)+0x9];
		unsigned offset3=(unsigned)t64Buf[0x40+(i*0x20)+0xa];
		unsigned offset4=(unsigned)t64Buf[0x40+(i*0x20)+0xb];
		unsigned offset=(offset4<<24) | (offset3)<<16 | (offset2<<8) | offset1;

		u8 *prgBuf=new u8[1024*1024];
		prgBuf[0]=startAddrLo;
		prgBuf[1]=startAddrHi;
		for(unsigned i=0; i<prgLen; i++) prgBuf[i+2]=t64Buf[offset+i];

		if(type==TCRT)
		{
			tapPos=tapecartBrowserAddPrg(prgBuf, prgLen, tapBuf, tapPos, prgName);
		} else {
			tapPos=prg2Tap (type, prgBuf, prgLen, tapBuf, tapPos, prgName, idxBuf);
		}

		delete[] prgBuf;
	}

	return tapPos;
}

// for mounting a tap when in tapecart mode
// construct tapecart image to notify user and return to basic
void TapUtils::mountTapecartTapLoader(u8 *tapecartBuffer, const char *filename)
{
	strcpy((char *)tapecartBuffer, "tapecartImage\015\012\032");

	tapecartBuffer[TCRT_VER]=0x01;
	tapecartBuffer[TCRT_VER+1]=0x00;

	tapecartBuffer[TCRT_OFFSET]=0x00;
	tapecartBuffer[TCRT_OFFSET+1]=0x00;

	u16 calladdr=0x0804;
	for(unsigned i=0; i<sizeof(tcrtTapStarter); i++)
	{
		tapecartBuffer[TCRT_DAT+i]=tcrtTapStarter[i];
	}

	for(unsigned i=0; filename[i] && i<120; i++)
	{
		tapecartBuffer[TCRT_DAT+0x3c+i]=toUpper(filename[i]);
	}

	tapecartBuffer[TCRT_DATALEN]=(sizeof(tcrtTapStarter))&0xff;
	tapecartBuffer[TCRT_DATALEN+1]=(sizeof(tcrtTapStarter))>>8;

	tapecartBuffer[TCRT_CALLADDR]=calladdr&0xff;
	tapecartBuffer[TCRT_CALLADDR+1]=calladdr>>8;

	const char *name="TAP LOADER";
	unsigned n;
	for(n=0; n<16 && name[n]; n++) tapecartBuffer[TCRT_NAME+n]=name[n];
	for(   ; n<16           ; n++) tapecartBuffer[TCRT_NAME+n]=0x20;

	tapecartBuffer[TCRT_FLAGS]=0x00;

	memset(tapecartBuffer+TCRT_LOADER, 0, 171);

	tapecartBuffer[TCRT_FLASHLEN]=(sizeof(tcrtTapStarter))&0xff;
	tapecartBuffer[TCRT_FLASHLEN+1]=(sizeof(tcrtTapStarter))>>8;
	tapecartBuffer[TCRT_FLASHLEN+2]=0x00;
	tapecartBuffer[TCRT_FLASHLEN+3]=0x00;
}

// data tapecart, eg. .sid file
unsigned TapUtils::data2Tapecart(u8 *dataBuf, unsigned dataLen,
			u8 *tapecartBuffer, unsigned tapecartBufSize, const char *name)
{
	strcpy((char *)tapecartBuffer, "tapecartImage\015\012\032");

	tapecartBuffer[TCRT_VER]=0x01;
	tapecartBuffer[TCRT_VER+1]=0x00;

	tapecartBuffer[TCRT_OFFSET]=0x00;
	tapecartBuffer[TCRT_OFFSET+1]=0x00;

	for(unsigned i=0; i<dataLen; i++) tapecartBuffer[TCRT_DAT+i]=dataBuf[i];

	tapecartBuffer[TCRT_DATALEN]=dataLen&0xff;
	tapecartBuffer[TCRT_DATALEN+1]=dataLen>>8;

	u16 calladdr=0;
	tapecartBuffer[TCRT_CALLADDR]=calladdr&0xff;
	tapecartBuffer[TCRT_CALLADDR+1]=calladdr>>8;

	unsigned n;
	for(n=0; n<16 && name[n]; n++) tapecartBuffer[TCRT_NAME+n]=name[n];
	for(   ; n<16           ; n++) tapecartBuffer[TCRT_NAME+n]=0x20;

	tapecartBuffer[TCRT_FLAGS]=0x00;

	memset(tapecartBuffer+TCRT_LOADER, 0, 171);

	tapecartBuffer[TCRT_FLASHLEN]=dataLen&0xff;
	tapecartBuffer[TCRT_FLASHLEN+1]=dataLen>>8;
	tapecartBuffer[TCRT_FLASHLEN+2]=0x00;
	tapecartBuffer[TCRT_FLASHLEN+3]=0x00;

	return TCRT_DAT+dataLen;
}

unsigned TapUtils::prg2Tapecart(const u8 *prgBuf, unsigned prgLen,
			u8 *tapecartBuffer, unsigned tapecartBufSize, const char *name)
{
	strcpy((char *)tapecartBuffer, "tapecartImage\015\012\032");

	tapecartBuffer[TCRT_VER]=0x01;
	tapecartBuffer[TCRT_VER+1]=0x00;

	tapecartBuffer[TCRT_OFFSET]=0x00;
	tapecartBuffer[TCRT_OFFSET+1]=0x00;

	u16 loadaddr=prgBuf[0] | (prgBuf[1]<<8);
	u16 calladdr=0xff40;
	if(loadaddr==0x0801)
	{
		for(unsigned i=0; i<sizeof(tcrtBasicStarter); i++)
		{
			tapecartBuffer[TCRT_DAT+2+i]=tcrtBasicStarter[i];
		}
		loadaddr-=sizeof(tcrtBasicStarter);
		calladdr=loadaddr;
		prgLen+=sizeof(tcrtBasicStarter);

		tapecartBuffer[TCRT_DAT]=loadaddr&0xff;
		tapecartBuffer[TCRT_DAT+1]=loadaddr>>8;
		unsigned idx=sizeof(tcrtBasicStarter)+2;
		for(unsigned i=0; i<prgLen-2; i++) tapecartBuffer[TCRT_DAT+idx+i]=prgBuf[i+2];
	} else {
		for(unsigned i=0; i<prgLen; i++) tapecartBuffer[TCRT_DAT+i]=prgBuf[i];
	}

	tapecartBuffer[TCRT_DATALEN]=prgLen&0xff;
	tapecartBuffer[TCRT_DATALEN+1]=prgLen>>8;

	tapecartBuffer[TCRT_CALLADDR]=calladdr&0xff;
	tapecartBuffer[TCRT_CALLADDR+1]=calladdr>>8;

	unsigned n;
	for(n=0; n<16 && name[n]; n++) tapecartBuffer[TCRT_NAME+n]=name[n];
	for(   ; n<16           ; n++) tapecartBuffer[TCRT_NAME+n]=0x20;

	tapecartBuffer[TCRT_FLAGS]=0x00;

	memset(tapecartBuffer+TCRT_LOADER, 0, 171);

	tapecartBuffer[TCRT_FLASHLEN]=prgLen&0xff;
	tapecartBuffer[TCRT_FLASHLEN+1]=prgLen>>8;
	tapecartBuffer[TCRT_FLASHLEN+2]=0x00;
	tapecartBuffer[TCRT_FLASHLEN+3]=0x00;

	return TCRT_DAT+prgLen;
}

unsigned TapUtils::tapecartLoader2Tap (const u8 *tapecartBuffer, unsigned tapecartSize,
	       u8 *tapBuf, unsigned tapPos, u8 *idxBuf)
{
	char name[17];
	for(int i=0; i<16; i++) name[i]=tapecartBuffer[TCRT_NAME+i];
	name[16]=0;

	if(tapecartBuffer[TCRT_FLAGS]&0x1) {
		tapPos=prg2Tap (TCRTLOADER, tapecartBuffer+TCRT_LOADER,
				0, tapBuf, tapPos, name, idxBuf);
	} else {
		tapPos=prg2Tap (TCRTLOADER, NULL,
				0, tapBuf, tapPos, name, idxBuf);
	}
	return tapPos;
}

unsigned TapUtils::tapecartBrowserInit(u8 *tapecartBuffer, const char *name)
{
	memset(tapecartBuffer, 0x00, TCRT_MAX_SIZE);

	strcpy((char *)tapecartBuffer, "tapecartImage\015\012\032");
	tapecartBuffer[TCRT_VER]=0x01;
	tapecartBuffer[TCRT_VER+1]=0x00;
	tapecartBuffer[TCRT_OFFSET]=0x00;
	tapecartBuffer[TCRT_OFFSET+1]=0x00;
	tapecartBuffer[TCRT_DATALEN]=(0x1000+sizeof(tcrtBrowser))&0xff;
	tapecartBuffer[TCRT_DATALEN+1]=(0x1000+sizeof(tcrtBrowser))>>8;

	u16 calladdr=0x2800;
	tapecartBuffer[TCRT_CALLADDR]=calladdr&0xff;
	tapecartBuffer[TCRT_CALLADDR+1]=calladdr>>8;

	unsigned n;
	for(n=0; n<16 && name[n]; n++) tapecartBuffer[TCRT_NAME+n]=toUpper(name[n]);
	for(   ; n<16           ; n++) tapecartBuffer[TCRT_NAME+n]=0x20;

	tapecartBuffer[TCRT_FLAGS]=0x00;
	memset(tapecartBuffer+TCRT_LOADER, 0, 171);

	// FS
	memset(tapecartBuffer+TCRT_DAT, 0xff, 0x1000);
	strcpy((char *)tapecartBuffer+TCRT_DAT, "\x02\x10TapcrtFileSys");

	// directory entry 1
	unsigned dirEntryBase=0xf;
	// filename
	for(unsigned i=0; i<16; i++) tapecartBuffer[TCRT_DAT+dirEntryBase+i]=0x20;
	const char *fsname="bROWSER-p1x3lNET";
	strcpy((char *)tapecartBuffer+TCRT_DAT+dirEntryBase, fsname);
	tapecartBuffer[TCRT_DAT+dirEntryBase+strlen(fsname)]=0x20;
	// type
	tapecartBuffer[TCRT_DAT+dirEntryBase+0x10]=0xfe;
	// start address
	tapecartBuffer[TCRT_DAT+dirEntryBase+0x11]=0x1000>>8;
	tapecartBuffer[TCRT_DAT+dirEntryBase+0x12]=0x1000>>16;
	// size
	tapecartBuffer[TCRT_DAT+dirEntryBase+0x13]=(16*1024)&0xff;
	tapecartBuffer[TCRT_DAT+dirEntryBase+0x14]=(16*1024)>>8;
	tapecartBuffer[TCRT_DAT+dirEntryBase+0x15]=(16*1024)>>16;
	// load address
	tapecartBuffer[TCRT_DAT+dirEntryBase+0x16]=0xff;
	tapecartBuffer[TCRT_DAT+dirEntryBase+0x17]=0xff;
	// bundle opts
	memset(tapecartBuffer+TCRT_DAT+dirEntryBase+0x18, 0, 8);

	// browser prg
	for(unsigned i=0; i<sizeof(tcrtBrowser); i++) tapecartBuffer[TCRT_DAT+0x1000+i]=tcrtBrowser[i];

	u32 flashlen=0x1000+(16*1024);
	tapecartBuffer[TCRT_FLASHLEN]=flashlen&0xff;
	tapecartBuffer[TCRT_FLASHLEN+1]=flashlen>>8;
	tapecartBuffer[TCRT_FLASHLEN+2]=flashlen>>16;
	tapecartBuffer[TCRT_FLASHLEN+3]=flashlen>>24;

	return TCRT_DAT+0x1000+(16*1024);
}

unsigned TapUtils::tapecartBrowserAddPrg(u8 *prgBuf, unsigned prgLen, u8 *tapecartBuffer, unsigned tapecartSize, const char *name)
{
	// find free dir entry
	unsigned dirEntryBase=0xf;
	unsigned dirEntry=0;
	for(dirEntry=0; tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x10]!=0xff; dirEntry++)
	{
		if(dirEntry>127) return tapecartSize;
	}

	// copy prg, data on 4096/0x1000 boundary, remove load addr
	u32 start=tapecartSize-TCRT_DAT;
	if(start&0xfff) start+=0x1000-(start&0xfff);
	if((TCRT_DAT+start+prgLen-2)>TCRT_MAX_SIZE) return tapecartSize;
	for(unsigned i=0; i<(prgLen-2); i++) tapecartBuffer[TCRT_DAT+start+i]=prgBuf[i+2];

	// filename
	unsigned n;
	for(n=0; n<16 && name[n]; n++) tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+n]=name[n];
	for(   ; n<16           ; n++) tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+n]=0x00;
	// type
	tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x10]=0x00;
	// start address
	tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x11]=start>>8;
	tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x12]=start>>16;
	// size
	tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x13]=prgLen&0xff;
	tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x14]=prgLen>>8;
	tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x15]=prgLen>>16;
	// load address
	tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x16]=prgBuf[0];
	tapecartBuffer[TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x17]=prgBuf[1];
	// bundle opts
	memset(tapecartBuffer+TCRT_DAT+dirEntryBase+(dirEntry*0x20)+0x18, 0, 8);

	// update tapecart flashlen
	u32 flashlen=start+prgLen-2;
	tapecartBuffer[TCRT_FLASHLEN]=flashlen&0xff;
	tapecartBuffer[TCRT_FLASHLEN+1]=flashlen>>8;
	tapecartBuffer[TCRT_FLASHLEN+2]=flashlen>>16;
	tapecartBuffer[TCRT_FLASHLEN+3]=flashlen>>24;

	return TCRT_DAT+start+prgLen-2;
}

#define SECTOR_SIZE   256
#define DIR_TRACK     18
#define DIR_ENTRY_SIZE 32
#define DIR_ENTRIES_PER_SECTOR 8

/* sectors per track (index = track-1) */
static const u8 d64_spt[35] = {
    21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,
    19,19,19,19,19,19,19,
    18,18,18,18,18,18,
    17,17,17,17,17
};

size_t TapUtils::d64_offset(int track, int sector)
{
    if (track < 1 || track > 35 || sector < 0 || sector >= d64_spt[track-1])
        return (size_t)-1;

    size_t off = 0;
    for (int t = 1; t < track; ++t)
        off += d64_spt[t-1] * SECTOR_SIZE;
    return off + (sector * SECTOR_SIZE);
}

const u8 *TapUtils::d64_get_sector(const u8 *img, int track, int sector)
{
    return img + d64_offset(track, sector);
}

void TapUtils::d64_sanitize(unsigned char *s, int n)
{
	for (int i=0; i<n; i++)
	{
		if(s[i]==0xa0) s[i]=0x20;
	}
}

unsigned TapUtils::d64_extract_prg(const u8 *img, int start_track, int start_sector, const char *name,
					tapConversions type, u8 *tapBuf, unsigned tapPos, u8 *idxBuf)
{
	size_t cap = 1024, used = 0;
	u8 *buf = (u8 *)malloc(cap);
	int trk = start_track, sec = start_sector;

	while (trk) {
		const u8 *sec_p = d64_get_sector(img, trk, sec);
		int next_trk = sec_p[0], next_sec = sec_p[1];
		size_t data_len = next_trk ? SECTOR_SIZE - 2 : sec_p[1] - 1;

		if (used + data_len > cap) {
			cap = cap * 2 + data_len;
			buf = (u8 *)realloc(buf, cap);
		}
		memcpy(buf + used, sec_p + 2, data_len);
		used += data_len;

		trk = next_trk;
		sec = next_sec;
	}

	if(type==TCRT)
	{
		tapPos=tapecartBrowserAddPrg(buf, used, tapBuf, tapPos, name);
	} else {
		tapPos=prg2Tap(type, buf, used, tapBuf, tapPos, name, idxBuf);
	}

	free(buf);

	return tapPos;
}

unsigned TapUtils::d642Tap (tapConversions type, const u8 *d64Buf, unsigned d64Len,
		u8 *tapBuf, unsigned tapPos, const char *name, u8 *idxBuf)
{
	int dir_trk = DIR_TRACK;
	int dir_sec = 1;
	while (dir_trk) {
		const u8 *dir = d64_get_sector(d64Buf, dir_trk, dir_sec);

		for (int i = 0; i < DIR_ENTRIES_PER_SECTOR; ++i) {
			const u8 *e = dir + i * DIR_ENTRY_SIZE;

			if ((e[2] & 0x07) != 0x02) continue;   /* PRG only */

			int trk = e[3], sec = e[4];
			//if (trk == 0) continue;

			char name[17];
			memcpy(name, e + 5, 16);
			name[16] = 0;
			d64_sanitize((unsigned char *)name, 16);

			tapPos=d64_extract_prg(d64Buf, trk, sec, name,
					type, tapBuf, tapPos, idxBuf);
		}

		dir_trk = dir[0];
		dir_sec = dir[1];   /* next directory sector link */
	}
	return tapPos;
}


void TapUtils::appendIdx (u8 *idxBuf, unsigned tapPos, const char *name)
{
	if(idxBuf==NULL) return;
	CString idxEntry;
	idxEntry.Format("0x%x %s\n", tapPos, name);
	strncat((char *)idxBuf, idxEntry, IDX_MAX_SIZE-strlen((char *)idxBuf)-1);
}

// kernal conversion code based on Thorsten Kattanek's command line converter
//  https://github.com/ThKattanek/c64_tap_tool

#define KERNAL_SHORT 0x30
#define KERNAL_MED   0x42
#define KERNAL_LONG  0x56

unsigned TapUtils::kernalPulse(char pulseLen, u8 *buf, int count) {
	for(int i=0; i<count; i++) buf[i]=pulseLen;
	return count;
}

unsigned TapUtils::kernalWriteByte(u8 *buf, u8 byte) {
	int i=0;
	i+=kernalPulse(KERNAL_LONG, buf+i, 1);
	i+=kernalPulse(KERNAL_MED, buf+i, 1);

	int parity=1;
	for(int j=0; j<8; j++) {
		if(byte&(1<<j)) {
			i+=kernalPulse(KERNAL_MED, buf+i, 1);
			i+=kernalPulse(KERNAL_SHORT, buf+i, 1);
			parity=!parity;
		} else {
			i+=kernalPulse(KERNAL_SHORT, buf+i, 1);
			i+=kernalPulse(KERNAL_MED, buf+i, 1);
		}
	}

	if(parity) {
		i+=kernalPulse(KERNAL_MED, buf+i, 1);
		i+=kernalPulse(KERNAL_SHORT, buf+i, 1);
	} else {
		i+=kernalPulse(KERNAL_SHORT, buf+i, 1);
		i+=kernalPulse(KERNAL_MED, buf+i, 1);
	}

	return i;
}

struct KERNAL_HEADER_BLOCK    // C64 Kernal Header TAP Block
{
    u8 header_type;
    u8 start_address_low;
    u8 start_address_high;
    u8 end_address_low;
    u8 end_address_high;
    char filename_displayed[16];
    char filename_not_displayed[171];
};

unsigned TapUtils::tapSpace (unsigned count, u8 *tapBuf)
{
	for(unsigned i=0; i<count; i++) tapBuf[i]=0x30;
	return count;
}

unsigned TapUtils::prg2Kernal (const u8 *prgBuf, unsigned prgLen, const u8 *headerDat, u8 *tapBuf, const char *name)
{
	unsigned tapPos=0;
	//tapPos+=kernalPulse(KERNAL_SHORT, tapBuf+tapPos, 27135);
	tapPos+=kernalPulse(KERNAL_SHORT, tapBuf+tapPos, 1500);
	for(u8 j=0x89; j>=0x81; j--) tapPos+=kernalWriteByte(tapBuf+tapPos, j);

	struct KERNAL_HEADER_BLOCK kernal_header;
	//kernal_header.header_type=0x01;
	kernal_header.header_type=0x03;

	kernal_header.start_address_low=prgBuf[0];
	kernal_header.start_address_high=prgBuf[1];
	unsigned start_address=(kernal_header.start_address_high<<8) | kernal_header.start_address_low;
	unsigned end_address=start_address+prgLen-2;
	kernal_header.end_address_low=end_address&0xff;;
	kernal_header.end_address_high=(end_address&0xff00)>>8;

	memset(kernal_header.filename_displayed, 0x20, sizeof(kernal_header.filename_displayed));
	memset(kernal_header.filename_not_displayed, 0x20, sizeof(kernal_header.filename_not_displayed));
	if(headerDat!=NULL)
	{
		for(int j=0; j<171; j++) kernal_header.filename_not_displayed[j]=headerDat[j];
	}

	//char filename_displayed[16];
	//asciiToUpper(name, filename_displayed, 16);
	asciiToUpper(name, kernal_header.filename_displayed, sizeof(kernal_header.filename_displayed));

	//const char *filename_not_displayed="";
	//strncpy(kernal_header.filename_displayed, filename_displayed, strlen(filename_displayed));
	//strncpy(kernal_header.filename_displayed, filename_displayed, sizeof(kernal_header.filename_displayed));
	//strncpy(kernal_header.filename_not_displayed, filename_not_displayed, strlen(filename_not_displayed));

	// kernal header
	u8 crc=0;
	for(unsigned j=0; j<sizeof(kernal_header); j++) {
		crc^=((u8 *)&kernal_header)[j];
		tapPos+=kernalWriteByte(tapBuf+tapPos, ((u8 *)&kernal_header)[j]);
	}
	tapPos+=kernalWriteByte(tapBuf+tapPos, crc);

	tapPos+=kernalPulse(KERNAL_LONG, tapBuf+tapPos, 1);
	tapPos+=kernalPulse(KERNAL_SHORT, tapBuf+tapPos, 1);

	tapPos+=kernalPulse(KERNAL_SHORT, tapBuf+tapPos, 79);
	for(u8 j=0x09; j>=0x01; j--) tapPos+=kernalWriteByte(tapBuf+tapPos, j);

	// kernal header backup
	crc=0;
	for(unsigned j=0; j<sizeof(kernal_header); j++) {
		crc^=((u8 *)&kernal_header)[j];
		tapPos+=kernalWriteByte(tapBuf+tapPos, ((u8 *)&kernal_header)[j]);
	}
	tapPos+=kernalWriteByte(tapBuf+tapPos, crc);

	tapPos+=kernalPulse(KERNAL_SHORT, tapBuf+tapPos, 5671);
	for(u8 j=0x89; j>=0x81; j--) tapPos+=kernalWriteByte(tapBuf+tapPos, j);

	// kernal data block
	crc=0;
	u8 byte;
	for(unsigned j=2; j<prgLen; j++) {
		byte=prgBuf[j];
		crc^=byte;
		tapPos+=kernalWriteByte(tapBuf+tapPos, byte);
	}
	tapPos+=kernalWriteByte(tapBuf+tapPos, crc);

	tapPos+=kernalPulse(KERNAL_LONG, tapBuf+tapPos, 1);
	tapPos+=kernalPulse(KERNAL_SHORT, tapBuf+tapPos, 1);

	tapPos+=kernalPulse(KERNAL_SHORT, tapBuf+tapPos, 79);
	for(u8 j=0x09; j>=0x01; j--) tapPos+=kernalWriteByte(tapBuf+tapPos, j);

	// kernal data block backup
	crc=0;
	for(unsigned j=2; j<prgLen; j++) {
		byte=prgBuf[j];
		crc^=byte;
		tapPos+=kernalWriteByte(tapBuf+tapPos, byte);
	}
	tapPos+=kernalWriteByte(tapBuf+tapPos, crc);

	return tapPos;
}


#define T64ZERO 0x1a
#define T64ONE  0x28

unsigned TapUtils::turboTape64WriteByte(u8 *buf, u8 byte)
{
	int i=0;
	for(int j=7; j>=0; j--) {
		if(byte&(1<<j)) buf[i++]=T64ONE;
		else buf[i++]=T64ZERO;
	}
	return i;
}

// https://web.archive.org/web/20220213152122/http://www.luigidifraia.com/c64/docs/tapeloaders.html#turbotape64
unsigned TapUtils::prg2TurboTape64 (const u8 *prgBuf, unsigned prgLen, u8 *tapBuf, const char *name)
{
	unsigned tapPos=0;
	// pilot byte
	for(int j=0; j<(256*4)+247; j++) tapPos+=turboTape64WriteByte(tapBuf+tapPos, 0x02);
	// sync train
	for(u8 j=0x09; j>=0x01; j--) tapPos+=turboTape64WriteByte(tapBuf+tapPos, j);

	// file type header
	tapPos+=turboTape64WriteByte(tapBuf+tapPos, 0x01);
	// header
	// load address
	tapPos+=turboTape64WriteByte(tapBuf+tapPos, prgBuf[0]);
	tapPos+=turboTape64WriteByte(tapBuf+tapPos, prgBuf[1]);
	// end address
	unsigned start_address=(prgBuf[1]<<8) | prgBuf[0];
	unsigned end_address=start_address+prgLen-2;
	tapPos+=turboTape64WriteByte(tapBuf+tapPos, end_address&0xff);
	tapPos+=turboTape64WriteByte(tapBuf+tapPos, (end_address&0xff00)>>8);
	// $00b0?
	tapPos+=turboTape64WriteByte(tapBuf+tapPos, 0x00);
	// filename
	char t64filename[16];
	for(int j=0; j<16; j++) t64filename[j]=0x20;
	asciiToUpper(name, t64filename, 16);
	//t64filename[strlen(t64filename)]=0x20;
	for(int j=0; j<16; j++) tapPos+=turboTape64WriteByte(tapBuf+tapPos, t64filename[j]);
	// payload
	for(int j=0; j<171; j++) tapPos+=turboTape64WriteByte(tapBuf+tapPos, 0x20);

	// pilot byte
	for(int j=0; j<256+247; j++) tapPos+=turboTape64WriteByte(tapBuf+tapPos, 0x02);
	// sync train
	for(u8 j=0x09; j>=0x01; j--) tapPos+=turboTape64WriteByte(tapBuf+tapPos, j);

	// file type data
	tapPos+=turboTape64WriteByte(tapBuf+tapPos, 0x00);
	// data
	u8 crc=0;
	u8 byte;
	for(unsigned j=2; j<prgLen; j++) {
		byte=prgBuf[j];
		crc^=byte;
		tapPos+=turboTape64WriteByte(tapBuf+tapPos, byte);
	}
	tapPos+=turboTape64WriteByte(tapBuf+tapPos, crc);

	// trailer
	for(int j=0; j<255; j++) tapPos+=turboTape64WriteByte(tapBuf+tapPos, 0x00);

	return tapPos;
}

void TapUtils::asciiToUpper (const char *src, char *dest, size_t dest_len)
{
    if (!src || !dest || dest_len == 0) return;

    size_t i = 0;
    //while (i < dest_len - 1 && *src) {
    while (i < dest_len && *src) {
        unsigned char c = (unsigned char)*src++;
        if (c >= 'a' && c <= 'z')
            c -= 0x20;          /* convert lowercase to uppercase */
        dest[i++] = (char)c;
    }
    //dest[i] = '\0';
}

u8 TapUtils::fromPetscii(u8 c) {
  if (c >= 0x41 && c <= 0x5a) c += 0x20;
  else if (c >= 0x61 && c <= 0x7a) c -= 0x20;
  return c;
}

u8 TapUtils::toPetscii(u8 c) {
  if (c >= 'a' && c <= 'z') c -= 0x20;
  else if (c >= 'A' && c <= 'Z') c += 0x20;
  return c;
}

u8 TapUtils::toUpper(u8 c) {
  if (c >= 'a' && c <= 'z') c -= 0x20;
  return c;
}

