//
// tapecart.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

/*
 Adapted from Tapecart by Ingo Korb
  https://github.com/ikorb/tapecart
 and Tapecart SD by Kim Jorgensen
  https://github.com/KimJorgensen/tapecart

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:
1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
SUCH DAMAGE.
*/

#include <circle/synchronize.h>
#include <string.h>
#include <strings.h>

#include "tapecart.h"

#define SREG_MAGIC_VALUE_LOADER  0xca65
#define SREG_MAGIC_VALUE_COMMAND 0xfce2

#define CONFIG_EXTMEM_SIZE (2*1024*1024)
#define CONFIG_EXTMEM_PAGE_SIZE 256
#define CONFIG_EXTMEM_ERASE_SIZE 4096

TapeCart::TapeCart (CTimer *pTimer, File *pFile, Tap *pTap, Status *pStatus, TapUtils *pTapUtils, NetUtils *pNetUtils, CActLED *pActLED)
	:
	m_pTimer (pTimer),
	m_pFile (pFile),
	m_pTap (pTap),
	m_pStatus (pStatus),
	m_pTapUtils (pTapUtils),
	m_pNetUtils (pNetUtils),
	m_pActLED (pActLED)
{
}

TapeCart::~TapeCart (void)
{
}

boolean TapeCart::Initialize (void)
{
	GPIOMotor=m_pTap->GPIOMotor;
	GPIORead =m_pTap->GPIORead;
	GPIOWrite=m_pTap->GPIOWrite;
	GPIOSense=m_pTap->GPIOSense;

	GPIOLEDMotor=m_pTap->GPIOLEDMotor;
	GPIOLEDRecord=m_pTap->GPIOLEDRecord;
	GPIOLEDPlay=m_pTap->GPIOLEDPlay;

	return TRUE;
}

unsigned TapeCart::mountTapecart (u8 *tapecart, const unsigned size)
{
	tcrtData=tapecart;
	tcrtSize=size;
	return 1;
}

//void TapeCart::setMode(tapecartMode_t mode)
//{
//	tapecartMode=mode;
//}

void TapeCart::tx_byte_common(u8 b)
{
	// wait until write is high or abort if motor on
	while(!GPIOWrite.Read()) if(!GPIOMotor.Read()) return;

	u64 timeNow=m_pTimer->GetClockTicks64();
	GPIOWrite.SetMode(GPIOModeOutput, FALSE);

	/* at 4us: set write to output, send bits 5+4 on sense+write */
	while(m_pTimer->GetClockTicks64()-timeNow < 4);
	GPIOSense.Write(!!(b & (1<<5)));
	GPIOWrite.Write(!!(b & (1<<4)));

	/* at 13us: send bits 7+6 on sense+write */
	while(m_pTimer->GetClockTicks64()-timeNow < 13);
	GPIOSense.Write(!!(b & (1<<7)));
	GPIOWrite.Write(!!(b & (1<<6)));

	/* at 22us: send bits 1+0 on sense+write */
	while(m_pTimer->GetClockTicks64()-timeNow < 22);
	GPIOSense.Write(!!(b & (1<<1)));
	GPIOWrite.Write(!!(b & (1<<0)));

	/* at 31us: send bits 3+2 on sense+write */
	while(m_pTimer->GetClockTicks64()-timeNow < 31);
	GPIOSense.Write(!!(b & (1<<3)));
	GPIOWrite.Write(!!(b & (1<<2)));

	/* at 41us: set write to input again */
	while(m_pTimer->GetClockTicks64()-timeNow < 41);
	GPIOWrite.SetMode(GPIOModeInput, FALSE);
	GPIOWrite.Write(HIGH);

	/* at 53us: wait until write is low to ensure we don't accidentally re-trigger immediately */
	while(m_pTimer->GetClockTicks64()-timeNow < 53);

	while(GPIOWrite.Read() && GPIOMotor.Read());
}

boolean TapeCart::tx_byte_loader(u8 b)
{
	if(!GPIOMotor.Read()) return TRUE;
	
	GPIOSense.Write(LOW);
	tx_byte_common(b);
	GPIOSense.Write(HIGH);

	return FALSE;
}

void TapeCart::loaderHandler()
{
	u16 dataofs=tcrtData[TCRT_OFFSET] | (tcrtData[TCRT_OFFSET+1]<<8);
	u16 datalen=tcrtData[TCRT_DATALEN] | (tcrtData[TCRT_DATALEN+1]<<8);
	u16 calladdr=tcrtData[TCRT_CALLADDR] | (tcrtData[TCRT_CALLADDR+1]<<8);
	u16 loadaddr=tcrtData[TCRT_DAT+dataofs] | (tcrtData[TCRT_DAT+dataofs+1]<<8);
	datalen-=2;
	u16 endaddr=loadaddr+datalen;

	if(m_pStatus->optionLogEnabled)
	{
		log.Format("tapecart loader: dataofs %x, datalen %x, calladdr %x, loadaddr %x\n", dataofs, datalen, calladdr, loadaddr);
		m_pStatus->appendLog(log);
	}

	GPIOLEDPlay.Write(HIGH);

	tx_byte_loader(calladdr&0xff);
	tx_byte_loader(calladdr>>8);
	tx_byte_loader(endaddr&0xff);
	tx_byte_loader(endaddr>>8);
	tx_byte_loader(loadaddr&0xff);
	tx_byte_loader(loadaddr>>8);

	unsigned idx=TCRT_DAT+dataofs+2;
	for(unsigned i=0; i<datalen; i++)
	{
		if(tx_byte_loader(tcrtData[idx+i])) break;
	}

	GPIOLEDPlay.Write(LOW);
}

boolean TapeCart::tx_byte_command(u8 b)
{
	if(!GPIOMotor.Read()) return TRUE;
	
	GPIOSense.Write(HIGH);
	tx_byte_common(b);
	GPIOSense.Write(LOW);

	return FALSE;
}

s16 TapeCart::c64_get_byte_error()
{
	GPIOSense.SetMode(GPIOModeOutput, FALSE);
	return -1;
}

s16 TapeCart::c64_get_byte()
{
	// signal ready
	GPIOSense.Write(HIGH);
	GPIOSense.SetMode(GPIOModeInput, FALSE);

	u8 v=0;
	for(u8 i=0; i<8; i++)
	{
		// wait for write high
		while(!GPIOWrite.Read()) if(!GPIOMotor.Read()) return c64_get_byte_error();
		// read bit from sense
		v = (v<<1) | !!GPIOSense.Read();
		// wait for write low, then loop
		while(GPIOWrite.Read()) if(!GPIOMotor.Read()) return c64_get_byte_error();
	}

	// wait for high to low edge on sense
	while(!GPIOSense.Read()) if(!GPIOMotor.Read()) return c64_get_byte_error();
	while( GPIOSense.Read()) if(!GPIOMotor.Read()) return c64_get_byte_error();

	// signal busy
	GPIOSense.SetMode(GPIOModeOutput, FALSE);
	GPIOSense.Write(LOW);
	return v;
}

boolean TapeCart::get_u8(u8 *result)
{
	s16 temp=c64_get_byte();
	if(temp<0) return TRUE;
	*result=(u8)temp&0xff;
	return FALSE;
}

boolean TapeCart::get_u16(u16 *result)
{
	s16 tmp;

	// low byte
	tmp=c64_get_byte();
	if(tmp<0) return TRUE;
	*result=tmp&0xff;

	// high byte
	tmp=c64_get_byte();
	if(tmp<0) return TRUE;
	*result|=(u16)(tmp&0xff)<<8;

	return FALSE;
}

boolean TapeCart::get_u24(u32 *result)
{
	s16 tmp;

	// low byte
	tmp=c64_get_byte();
	if(tmp<0) return TRUE;
	*result=tmp&0xff;

	// middle byte
	tmp=c64_get_byte();
	if(tmp<0) return TRUE;
	*result|=(u32)(tmp&0xff)<<8;

	// high byte
	tmp=c64_get_byte();
	if(tmp<0) return TRUE;
	*result|=(u32)(tmp&0xff)<<16;

	return FALSE;
}

boolean TapeCart::c64_send_byte(u8 v)
{
	// signal ready
	GPIOSense.Write(HIGH);

	// wait for write high
	while(!GPIOWrite.Read()) if(!GPIOMotor.Read()) return TRUE;

	for(u8 i=0; i<8; i++)
	{
		if(!GPIOMotor.Read()) return TRUE;

		// send on 1->0 transition
		while(GPIOWrite.Read()) if(!GPIOMotor.Read()) return TRUE;

		if(v&0x80) GPIOSense.Write(HIGH);
		else GPIOSense.Write(LOW);

		v <<= 1;

		// wait for 0->1 transition
		while(!GPIOWrite.Read()) if(!GPIOMotor.Read()) return TRUE;
	}

	// wait until write is low again
	while(GPIOWrite.Read()) if(!GPIOMotor.Read()) return TRUE;

	// signal busy
	GPIOSense.Write(LOW);

	return FALSE;
}

boolean TapeCart::c64_send_u16(u16 v)
{
	if (c64_send_byte(v & 0xff)) return true;
	return c64_send_byte((v >> 8) & 0xff);
}

boolean TapeCart::c64_send_u24(u32 v)
{
	if (c64_send_byte(v & 0xff))        return true;
	if (c64_send_byte((v >> 8) & 0xff)) return true;
	return c64_send_byte((v >> 16) & 0xff);
}

boolean TapeCart::c64_send_u32(u32 v)
{
	if (c64_send_byte(v & 0xff))        return true;
	if (c64_send_byte((v >> 8) & 0xff)) return true;
	if (c64_send_byte((v >> 16) & 0xff)) return true;
	return c64_send_byte((v >> 24) & 0xff);
}

u32 TapeCart::crc32b(u8 *message, unsigned len)
{
   unsigned i, byte, crc, mask;
   int j;

   crc = 0xFFFFFFFF;
   for (i=0; i<len; i++) {
      byte = message[i];            // Get next byte.
      crc = crc ^ byte;
      for (j = 7; j >= 0; j--) {    // Do eight times.
         mask = -(crc & 1);
         crc = (crc >> 1) ^ (0xEDB88320 & mask);
      }
   }
   return ~crc;
}

void TapeCart::crc32_flash(void)
{
  /* parameters:
   *   3   address
   *   3   length
   * reply:
   *   4   crc32
   */

	u32 addr, len, crc32;

	if(get_u24(&addr)) return;
	if(get_u24(&len)) return;

	crc32=crc32b(tcrtData+TCRT_DAT+addr, len);

	if(m_pStatus->optionLogEnabled)
	{
		log.Format("crc32_flash: addr %x, len %x, crc32 %x\n", addr, len, crc32);
		m_pStatus->appendLog(log);
	}

	c64_send_u32(crc32);
}

void TapeCart::erase_flash_64k(void)
{
  /* parameters:
   *   3 address
   */

	u32 addr;
	if(get_u24(&addr)) return;

	if(m_pStatus->optionLogEnabled)
	{
		//CString log;
		log.Format("erase_flash_64k: addr %x\n", addr);
		m_pStatus->appendLog(log);
	}

	addr&=(u32)~0xffff;
	for(unsigned i=0; i<0x10000; i++)
	{
		tcrtData[TCRT_DAT+addr+i]=0x00;
	}
	writeTapecartFile();
}

void TapeCart::erase_flash_block(void)
{
  /* parameters:
   *   3 address
   */

	u32 addr;
	if(get_u24(&addr)) return;

	if(m_pStatus->optionLogEnabled)
	{
		//CString log;
		log.Format("erase_flash_block: addr %x\n", addr);
		m_pStatus->appendLog(log);
	}

	for(unsigned i=0; i<CONFIG_EXTMEM_ERASE_SIZE; i++)
	{
		tcrtData[TCRT_DAT+addr+i]=0x00;
	}
	writeTapecartFile();
}

void TapeCart::do_dir_lookup(void)
{
  /* parameters:
   *   n byte name
   * reply:
   *   1 byte result (0 if name was found, non-0 otherwise)
   *  if result is 0:
   *   m byte data
   */

	u8 dir_namebuf[16+1];

	/* read name from C64 */
	for(u8 i=0; i<dir_name_len; i++)
	{
		if(get_u8(dir_namebuf+i)) return;
	}

	if(m_pStatus->optionLogEnabled)
	{
		//CString log;
		log.Format("do_dir_lookup: %s\n", dir_namebuf);
		m_pStatus->appendLog(log);
	}

	/* find name in flash */
	bool	skip = true;
	u16	entry = 0;

	unsigned idx=TCRT_DAT+dir_base;
	while (skip && entry < dir_entries) {
		skip = false;

		/* check for match in the current dir entry */
		for (u8 i = 0; i < dir_name_len; i++) {
			u8 b = tcrtData[idx++];

			/* check for match */
			if (dir_namebuf[i] != b) skip = true;
		}

		/* send "found it" mark if needed */
		if (!skip) if (c64_send_byte(0)) return;

		/* skip or send data */
		for (u8 i = 0; i < dir_data_len; i++) {
			u8 b = tcrtData[idx++];
			if (!skip) if (c64_send_byte(b)) return;
		}

		entry++;
	}

	if (skip) {
		/* not found */
		c64_send_byte(1);
	}
}

void TapeCart::read_capabilities() {
  /* no parameters
   * reply:
   *  4 extended capability flags
   */

	if(c64_send_u16(0)) return;
	c64_send_u16(0);
}

void TapeCart::read_debugflags() {
	c64_send_u16(debug_flags);
}

void TapeCart::read_deviceinfo() {
	const char *device="PITAP";
	int len=strlen(device);
	for(int i=0; i<len; i++)
		if(c64_send_byte((u8)device[i])) return;
	c64_send_byte(0);
}

void TapeCart::read_devicesizes()
{
  /* no parameters
   * reply:
   *   3 extmem total size (in byte)
   *   2 extmem page  size (in byte)
   *   2 extmem erase size (in pages - if 0, direct byte write supported)
   */

  if (c64_send_u24(CONFIG_EXTMEM_SIZE)) return;

  if (c64_send_u16(CONFIG_EXTMEM_PAGE_SIZE)) return;

  c64_send_u16(CONFIG_EXTMEM_ERASE_SIZE / CONFIG_EXTMEM_PAGE_SIZE);
}

void TapeCart::read_flash(boolean fast) {
  /* parameters:
   *   3   address
   *   2   length
   * reply:
   *   len data
   */

	u32 addr, end;
	u16 len;

	if(get_u24(&addr)) return;
	if(get_u16(&len)) return;

	end=addr+len;

	if(m_pStatus->optionLogEnabled)
	{
		//CString log;
		log.Format("read flash addr:%x len:%x\n", addr, len);
		m_pStatus->appendLog(log);
	}

	while(addr<end)
	{
		if(fast)
		{
			tx_byte_command(tcrtData[TCRT_DAT+addr]);
		} else {
			if(c64_send_byte(tcrtData[TCRT_DAT+addr])) return;
		}
		addr++;
	}
}

void TapeCart::read_loader(void) {
  /* no parameters
   * reply:
   *   171 loader binary
   */

	if(tcrtData[TCRT_FLAGS]&0x1)
	{
		for (unsigned i=0; i<(TCRT_FLASHLEN-TCRT_LOADER); i++)
			if(c64_send_byte(tcrtData[TCRT_LOADER+i])) return;
	} else {
		for (unsigned i=0; i<(TCRT_FLASHLEN-TCRT_LOADER); i++)
			//if(c64_send_byte(tempLoader[i])) return;
			if(c64_send_byte(m_pTapUtils->tcrtLoader[i])) return;
	}
}

void TapeCart::read_loadinfo(void) {
  /* no parameters
   * reply:
   *   2 data address
   *   2 data length
   *   2 call address
   *  16 file name
   */

	if(c64_send_byte(tcrtData[TCRT_OFFSET])) return;
	if(c64_send_byte(tcrtData[TCRT_OFFSET+1])) return;
	if(c64_send_byte(tcrtData[TCRT_DATALEN])) return;
	if(c64_send_byte(tcrtData[TCRT_DATALEN+1])) return;
	if(c64_send_byte(tcrtData[TCRT_CALLADDR])) return;
	if(c64_send_byte(tcrtData[TCRT_CALLADDR+1])) return;

	for(unsigned i=0; i<16; i++)
	{
		if(c64_send_byte(tcrtData[TCRT_NAME+i])) return;
	}
}

void TapeCart::set_dirparams()
{
  /* parameters:
   *   3 byte directory start address in flash
   *   2 byte number of dir entries
   *   1 byte name length
   *   1 byte data length
   * no reply
   */

	if(get_u24(&dir_base)) return;

	if(get_u16(&dir_entries)) return;

	if(get_u8(&dir_name_len)) return;
	if(dir_name_len>16) dir_name_len=16;

	if(get_u8(&dir_data_len)) return;
}

void TapeCart::sd_open_dir()
{
  /* parameters:
   *   16 byte dir name (null-terminated if shorter than 16 bytes)
   * reply:
   *   16 byte current dir name (null-terminated if shorter than 16 bytes)
   */

	u8 dir_namebuf[16+1];
	unsigned i;

	/* read name from C64 */
	for (i = 0; i < 16; i++) {
		if (get_u8(dir_namebuf + i)) return;
		dir_namebuf[i]=m_pTapUtils->fromPetscii(dir_namebuf[i]);
	}
	dir_namebuf[16] = 0;
	char *name=(char *)dir_namebuf;
	FILINFO *dirList=new FILINFO[maxFiles];

	char cwd[1024];
	strncpy(cwd, currentDir, 1024);

	unsigned clen=strlen(cwd);
	if(cwd[clen-1]=='/') cwd[clen-1]=0;

	if(!strcmp((char *)name, "."))
	{
		name[0]=0;
	} else if(!strcmp((char *)name, ".."))
	{
		for(i=strlen(cwd); i>0; i--)
		{
			if(cwd[i-1]=='/')
			{
				cwd[i-1]=0;
				break;
			}
		}
		name[0]=0;
	} else if(!strcmp((char *)name, "/"))
	{
		strcpy(cwd, "");
		name[0]=0;
	} else if(name[12]=='#')
	{
		// convert truncated display name to real name
		unsigned dirNum;
		dirNum=m_pFile->readDir(dirList, maxFiles, currentDir);
		unsigned idx=((name[13]-0x30)*100)+((name[14]-0x30)*10)+(name[15]-0x30);
		if(idx<dirNum) name=dirList[idx].fname;
	}

	currentDir.Format("%s/%s", cwd, (char *)name);

	delete[] dirList;

	if(!(m_pFile->fileStat((const char *)currentDir))) currentDir="/";

	for(i=0; i<16 && currentDir[i]; i++) dir_namebuf[i]=currentDir[i];
	for(   ; i<16                 ; i++) dir_namebuf[i]=0;

	for (i=0; i<16; i++) if(c64_send_byte(dir_namebuf[i])) return;
}

void TapeCart::sd_read_dir()
{
  /* parameters:
   *   2 byte max number of files (0: no limit)
   * reply:
   *   1 byte file type (0: end of dir, see file_t)
   *   3 byte file size
   *  16 byte filename (null-terminated if shorter than 16 bytes)
   */

	unsigned i;

	if(get_u16(&maxFiles)) return;

	if(maxFiles==0) maxFiles=1000;
	maxFiles-=1; // space for FILE_NONE at end

	unsigned dirNum;
	FILINFO *dirList=new FILINFO[maxFiles];
	dirNum=m_pFile->readDir(dirList, maxFiles, currentDir);
	CString displayName;

	// dirs
	for(i=0; i<dirNum; i++)
	{
		// file type
		if(dirList[i].fattrib&AM_DIR) tx_byte_command(FILE_DIR);
		else continue;
		// file size
		tx_byte_command(dirList[i].fsize);
		tx_byte_command(dirList[i].fsize>>8);
		tx_byte_command(dirList[i].fsize>>16);
		// file name
		if(strlen(dirList[i].fname)>16)
		{
			dirList[i].fname[12]=0;
			displayName.Format("%s#%03d", dirList[i].fname, i);
		}
		else displayName.Format("%s", dirList[i].fname);
		unsigned j;
		for(j=0; j<16 && displayName[j]; j++) tx_byte_command(m_pTapUtils->toPetscii(displayName[j]));
		for(   ; j<16                  ; j++) tx_byte_command(0x0);
	}

	// files
	for(i=0; i<dirNum; i++)
	{
		// file type
		if(!(dirList[i].fattrib&AM_DIR))
		{
			char *name=dirList[i].fname;
			int len=strlen(name);
			int type=FILE_PRG;
			if(len>4 && !strcasecmp(name+(len-4), ".sid")) type=FILE_SID;
			if(len>5 && !strcasecmp(name+(len-5), ".tcrt")) type=FILE_TCRT;
			tx_byte_command(type);
		} else continue;
		// file size
		tx_byte_command(dirList[i].fsize);
		tx_byte_command(dirList[i].fsize>>8);
		tx_byte_command(dirList[i].fsize>>16);
		// file name
		if(strlen(dirList[i].fname)>16)
		{
			dirList[i].fname[12]=0;
			displayName.Format("%s#%03d", dirList[i].fname, i);
		}
		else displayName.Format("%s", dirList[i].fname);
		unsigned j;
		for(j=0; j<16 && displayName[j]; j++) tx_byte_command(m_pTapUtils->toPetscii(displayName[j]));
		for(   ; j<16                  ; j++) tx_byte_command(0x0);
	}

	delete[] dirList;

	// end of dir
	tx_byte_command(FILE_NONE);
	tx_byte_command(0x0);
	tx_byte_command(0x0);
	tx_byte_command(0x0);
	for(i=0; i<16; i++) tx_byte_command(0x0);
}

void TapeCart::sd_select_file()
{
  /* parameters:
   *   16 byte dir name (null-terminated if shorter than 16 bytes)
   * no reply
   */

	u8 dir_namebuf[16+1];
	u8 i;

	/* read name from C64 */
	for (i = 0; i < 16; i++) {
		if (get_u8(dir_namebuf + i)) return;
		dir_namebuf[i]=m_pTapUtils->fromPetscii(dir_namebuf[i]);
	}
	dir_namebuf[16] = 0;

	char *name=(char *)dir_namebuf;
	FILINFO *dirList=new FILINFO[maxFiles];
	if(name[12]=='#')
	{
		// convert truncated display name to real name
		unsigned dirNum;
		dirNum=m_pFile->readDir(dirList, maxFiles, currentDir);
		unsigned idx=((name[13]-0x30)*100)+((name[14]-0x30)*10)+(name[15]-0x30);
		if(idx<dirNum) name=dirList[idx].fname;
	}

	m_pStatus->tcrtLock.Acquire();
	m_pStatus->currentTapecart=currentDir;
	m_pStatus->currentTapecart+="/";
	m_pStatus->currentTapecart+=name;
	m_pStatus->tapecartFileChanged=TRUE;
	m_pStatus->tcrtLock.Release();
	delete[] dirList;

	// wait for kernel.cpp to mount file
	boolean exit=FALSE;
	while(!exit)
	{
		m_pTimer->SimpleusDelay(100000);
		m_pStatus->tcrtLock.Acquire();
		if(m_pStatus->tapecartFileChanged==FALSE) exit=TRUE;
		m_pStatus->tcrtLock.Release();
	}
}

void TapeCart::setLED(boolean v)
{
	if(v) m_pActLED->On();
	else m_pActLED->Off();
}

void TapeCart::write_debugflags() {
	get_u16(&debug_flags);
}

void TapeCart::write_flash(void)
{
  /* parameters:
   *   3   address
   *   2   length
   * len   data
   */

	u16 len;
	u32 addr, end;

	if(get_u24(&addr)) return;
	if(get_u16(&len)) return;
	end = addr + len;

	if(m_pStatus->optionLogEnabled)
	{
		//CString log;
		log.Format("write_flash: addr %x, len:%x\n", addr, len);
		m_pStatus->appendLog(log);
	}

	while(addr!=end)
	{
		u8 tmp;
		if(get_u8(&tmp)) return;
		tcrtData[TCRT_DAT+addr]=tmp;
		addr++;
	}

	writeTapecartFile();
}

void TapeCart::write_loader(void)
{
  /* parameters:
   *   171 loader code
   * no reply
   */

	u8 v;
	for(unsigned i=0; i<171; i++)
	{
		if(get_u8(&v)) return;
		tcrtData[TCRT_LOADER+i]=v;
	}
	writeTapecartFile();
}

void TapeCart::write_loadinfo(void)
{
  /* parameters:
   *   2 data address
   *   2 data length
   *   2 call address
   *  16 file name
   * no reply
   */

	u16 val;

	if(get_u16(&val)) return;
	tcrtData[TCRT_OFFSET]=val&0xff;
	tcrtData[TCRT_OFFSET+1]=val>>8;

	if(get_u16(&val)) return;
	tcrtData[TCRT_DATALEN]=val&0xff;
	tcrtData[TCRT_DATALEN+1]=val>>8;

	if(get_u16(&val)) return;
	tcrtData[TCRT_CALLADDR]=val&0xff;
	tcrtData[TCRT_CALLADDR+1]=val>>8;

	u8 v;
	for(unsigned i=0; i<16; i++)
	{
		if(get_u8(&v)) return;
		tcrtData[TCRT_NAME+i]=v;
	}
	writeTapecartFile();
}

void TapeCart::writeTapecartFile()
{
	if(m_pStatus->tapecartDynamic) return;
	m_pFile->writeFile(tcrtData, tcrtSize, m_pStatus->currentTapecart);
}

void TapeCart::cmd_dnsResolve()
{
  /* CMD_DNS_RESOLVE
   * parameters:
   *   DNS name - null-terminated string
   *    can be a host name or dotted quad IP address string
   * reply:
   *   1 result 0 on error
   *   4 IP address
   */
	u8 name[255];
	unsigned i;
	for(i=0; i<254; i++)
	{
		u8 c;
		if(get_u8(&c)) return;
		name[i]=c;
		if(c=='\0') break;
	}
	if(i==254) name[i]='\0';

	m_pNetUtils->jobLock.Acquire();
	m_pNetUtils->pendingJob=NET_DNS_RESOLVE;
	m_pNetUtils->DNSResolveName=(char *)name;
	m_pNetUtils->jobLock.Release();

	boolean exit=FALSE;
	while(!exit)
	{
		DataSyncBarrier();
		if(m_pNetUtils->pendingJob==NET_NONE) exit=TRUE;
	}

	u32 ip=m_pNetUtils->DNSResolveResult;
	if(ip==0) c64_send_byte(0);
	else c64_send_byte(1);

	c64_send_byte(ip);
	c64_send_byte(ip>>8);
	c64_send_byte(ip>>16);
	c64_send_byte(ip>>24);
}

void TapeCart::cmd_httpGet()
{
  /* CMD_HTTP_GET
   * parameters:
   *   4 ip address - host address
   *    set to zero if the host is included as a string in the next field
   *   n host name/url path - null-terminated string
   *    if the ip address field is zeroes provide the host name or dotted quad IP address string followed by the path
   *    if ip already supplied then just provide the path
   *   2 server port
   *   3 offset
   *   2 maximum content length to return
   *   1 flags
   *     bit 0 - transfer protocol: 0=use the normal command mode 1-bit protocol
   *                                1=use the fast 2-bit protocol as used by READ_FLASH_FAST
   *   headers?
   * reply:
   *   2 http status code
   *   3 received content length
   *   n content
   */

	// TODO
}

void TapeCart::cmd_httpMount()
{
  /* CMD_HTTP_MOUNT
   * parameters:
   *   n host name/url path - null-terminated string
   * reply:
   *   1 result 0 on error
   */

	// TODO
}

void TapeCart::cmd_netStatus()
{
  /* CMD_NET_STATUS
   * no parameters
   * reply:
   *   1 1 if connected
   *   4 ip address
   *   4 net mask
   *   4 gateway
   *   4 dns server
   *   4 broadcast address
   *  16 hostname
   */

	m_pNetUtils->jobLock.Acquire();
	m_pNetUtils->pendingJob=NET_STATUS;
	m_pNetUtils->jobLock.Release();
	boolean exit=FALSE;
	while(!exit)
	{
		DataSyncBarrier();
		if(m_pNetUtils->pendingJob==NET_NONE) exit=TRUE;
	}

	//connected
	if(m_pNetUtils->connected) c64_send_byte(1);
	else c64_send_byte(0);

	// ip addr
	u32 ipAddr=m_pNetUtils->ipAddr;
	c64_send_byte(ipAddr);
	c64_send_byte(ipAddr>>8);
	c64_send_byte(ipAddr>>16);
	c64_send_byte(ipAddr>>24);

	// netMask
	u32 netMask=m_pNetUtils->netMask;
	c64_send_byte(netMask);
	c64_send_byte(netMask>>8);
	c64_send_byte(netMask>>16);
	c64_send_byte(netMask>>24);

	// gateway
	u32 gateway=m_pNetUtils->gateway;
	c64_send_byte(gateway);
	c64_send_byte(gateway>>8);
	c64_send_byte(gateway>>16);
	c64_send_byte(gateway>>24);

	// dnsServer
	u32 dnsServer=m_pNetUtils->dnsServer;
	c64_send_byte(dnsServer);
	c64_send_byte(dnsServer>>8);
	c64_send_byte(dnsServer>>16);
	c64_send_byte(dnsServer>>24);

	// broadcastAddr
	u32 broadcastAddr=m_pNetUtils->broadcastAddr;
	c64_send_byte(broadcastAddr);
	c64_send_byte(broadcastAddr>>8);
	c64_send_byte(broadcastAddr>>16);
	c64_send_byte(broadcastAddr>>24);

	// hostname
	const char *hostname=m_pNetUtils->hostname;
	int i;
	for(i=0; hostname[i] && i<16; i++) c64_send_byte((u8)hostname[i]);
	for(   ;                i<16; i++) c64_send_byte(0x20);
}

void TapeCart::cmd_ntpSync()
{
  /* CMD_NTP_SYNC
   * no parameters
   * reply:
   *   1 returns 0 on error
   */
	/*m_pNetUtils->ntpLock.Acquire();
	m_pNetUtils->doNTPUpdate=TRUE;
	m_pNetUtils->ntpLock.Release();

	boolean exit=FALSE;
	while(!exit)
	{
		m_pTimer->SimpleusDelay(100000);
		m_pNetUtils->ntpLock.Acquire();
		if(m_pNetUtils->doNTPUpdate==FALSE) exit=TRUE;
		m_pNetUtils->ntpLock.Release();
	}*/

	m_pNetUtils->jobLock.Acquire();
	m_pNetUtils->pendingJob=NET_NTP_SYNC;
	m_pNetUtils->jobLock.Release();

	boolean exit=FALSE;
	while(!exit)
	{
		DataSyncBarrier();
		if(m_pNetUtils->pendingJob==NET_NONE) exit=TRUE;
	}

	//while(m_pNetUtils->pendingJob!=NET_NONE);

	if(m_pNetUtils->lastNTPSync!=0) c64_send_byte(1);
	else c64_send_byte(0);
}

void TapeCart::cmd_setTimeZone()
{
  /* CMD_SET_TIMEZONE
   * parameters:
   *   1 signed timezone in quarter hours from UTC
   *     eg. UTC+1=4, UTC-8=-32
   * no reply
   */
	u8 tz;
	if(get_u8(&tz)) return;
	m_pNetUtils->setTimeZone(((s8)tz)*15);
}

void TapeCart::cmd_getTime()
{
  /* CMD_GET_TIME
   * no parameters
   * reply (little-endian):
   *   2 year (YYYY)
   *   1 century
   *   1 year%100 (YY)
   *   1 month (1-12)
   *   1 day of month
   *   1 hours
   *   1 minutes
   *   1 seconds
   *   1 hours (bcd, bit 7 0=am, 1=pm)
   *   1 minutes (bcd)
   *   1 seconds (bcd)
   *   1 weekday (0=Sunday)
   *   4 seconds since 1970 (unsigned)
   *   1 sync status
   *     0-254 hours since last ntp sync
   *     255 not synced
   */

	u8 timeData[18];
	m_pNetUtils->tcGetTime(timeData);
	for(unsigned i=0; i<18; i++) c64_send_byte(timeData[i]);
}

void TapeCart::commandHandler()
{
	// accept commands while motor is off
	while (GPIOMotor.Read())
	{
		if(debug_flags & DEBUGFLAG_SEND_CMDOK)
		{
			if(c64_send_byte('O')) return;
			if(c64_send_byte('K')) return;
		}

		static u8 cmd;

		GPIOLEDPlay.Write(LOW);
		GPIOLEDRecord.Write(LOW);

		if(get_u8(&cmd)) break;

		GPIOLEDPlay.Write(HIGH);

		// curl pitap.lan/getlog for cmd log
		if(m_pStatus->optionLogEnabled || (debug_flags & DEBUGFLAG_BLINK_COMMAND))
		{
			//CString log;
			log.Format("tcrt cmd: %x\n", cmd);
			m_pStatus->appendLog(log);
		}

		switch(cmd)
		{
			default:
				break;
			case CMD_EXIT:
				return;
			case CMD_READ_DEVICEINFO:
				read_deviceinfo();
				break;
			case CMD_READ_DEVICESIZES:
				read_devicesizes();
				break;
			case CMD_READ_CAPABILITIES:
				read_capabilities();
				break;

			case CMD_READ_FLASH:
				read_flash(FALSE);
				break;
			case CMD_READ_FLASH_FAST:
				read_flash(TRUE);
				break;
			case CMD_WRITE_FLASH:
				GPIOLEDRecord.Write(HIGH);
				write_flash();
				break;
			case CMD_WRITE_FLASH_FAST:
				// not implemented
				break;
			case CMD_ERASE_FLASH_64K:
				GPIOLEDRecord.Write(HIGH);
				erase_flash_64k();
				break;
			case CMD_ERASE_FLASH_BLOCK:
				GPIOLEDRecord.Write(HIGH);
				erase_flash_block();
				break;
			case CMD_CRC32_FLASH:
				GPIOLEDRecord.Write(HIGH);
				crc32_flash();
				break;

			case CMD_READ_LOADER:
				read_loader();
				break;
			case CMD_READ_LOADINFO:
				read_loadinfo();
				break;
			case CMD_WRITE_LOADER:
				GPIOLEDRecord.Write(HIGH);
				write_loader();
				break;
			case CMD_WRITE_LOADINFO:
				GPIOLEDRecord.Write(HIGH);
				write_loadinfo();
				break;

			case CMD_LED_OFF:
				setLED(FALSE);
				break;
			case CMD_LED_ON:
				setLED(TRUE);
				break;
			case CMD_READ_DEBUGFLAGS:
				read_debugflags();
				break;
			case CMD_WRITE_DEBUGFLAGS:
				write_debugflags();
				break;

			case CMD_DIR_SETPARAMS:
				set_dirparams();
				break;
			case CMD_DIR_LOOKUP:
				do_dir_lookup();
				break;

			case CMD_SD_OPEN_DIR:
				sd_open_dir();
				break;
			case CMD_SD_READ_DIR_FAST:
				sd_read_dir();
				break;
			case CMD_SD_SELECT_FILE:
				sd_select_file();
				break;

			case CMD_NET_STATUS:
				cmd_netStatus();
				break;
			case CMD_NTP_SYNC:
				cmd_ntpSync();
				break;
			case CMD_SET_TIMEZONE:
				cmd_setTimeZone();
				break;
			case CMD_GET_TIME:
				cmd_getTime();
				break;
			case CMD_DNS_RESOLVE:
				cmd_dnsResolve();
				break;

			case CMD_RAMEXEC:
				break;
		}
	}
}

void TapeCart::c64CommandHandler()
{
	GPIORead.Write(LOW);
	GPIOSense.Write(HIGH);

	//CString log;

	// wait until write is high or motor on
	while(GPIOMotor.Read() && !GPIOWrite.Read());
	if(!GPIOMotor.Read()) return;

	/* send pulses until write is low or motor on */
	boolean first_half=FALSE;
	while(GPIOMotor.Read() && GPIOWrite.Read())
	{
		if(!GPIOMotor.Read()) return;
		if(first_half)
		{
			m_pTimer->SimpleusDelay((0x18*8)/2);
			GPIORead.Write(HIGH);
		} else {
			m_pTimer->SimpleusDelay((0x18*8)/2);
			GPIORead.Write(LOW);
		}
		first_half=!first_half;
	}
	
	GPIORead.Write(LOW);

	commandHandler();

	GPIOLEDPlay.Write(LOW);
	GPIOLEDRecord.Write(LOW);
	GPIOSense.Write(HIGH);
}

/*void TapeCart::sendPulse(void)
{
	static boolean first_half=FALSE;
	if(first_half)
	{
		m_pTimer->SimpleusDelay((0x18*8)/2);
		GPIORead.Write(HIGH);
	} else {
		m_pTimer->SimpleusDelay((0x18*8)/2);
		GPIORead.Write(LOW);
	}
	first_half=!first_half;
}*/

void TapeCart::checkShiftreg (void)
{
	static boolean lastMotor=FALSE;

	boolean motorOn=!GPIOMotor.Read();

	// if motor turns on check shiftreg for mode change
	if(motorOn!=lastMotor)
	{
		if(motorOn)
		{
			shiftreg=(shiftreg<<1) | !!GPIOWrite.Read();

			switch (shiftreg) {
				case SREG_MAGIC_VALUE_LOADER:
					tapecartMode=MODE_LOADER;
					break;
				case SREG_MAGIC_VALUE_COMMAND:
					tapecartMode=MODE_C64COMMAND;
					break;
				default:
					tapecartMode=MODE_STREAM;
					break;
			}
		}
		lastMotor=motorOn;
	}
}

void TapeCart::hwInterface (void)
{
	// enable sense for 3s, disable for 200ms
	if(!m_pTap->isBusy() && m_pStatus->optionShiftregEnableSense)
	{
		static u64 shiftregSenseOnTime=0;
		u64 timeNow=m_pTimer->GetClockTicks64();
		if(timeNow-shiftregSenseOnTime>(3200*1000))
		{
			GPIOSense.Write(LOW); // enable sense
			shiftregSenseOnTime=timeNow;
		}
		if(timeNow-shiftregSenseOnTime>(3000*1000))
		{
			GPIOSense.Write(HIGH); // disable sense
		}
	}
	checkShiftreg();

	switch (tapecartMode) {
		case MODE_LOADER:
			GPIOSense.Write(HIGH);
			// delay slightly to ensure the C64 has turned the motor signal off
			m_pTimer->SimpleusDelay(100*1000);
			GPIOLEDMotor.Write(!GPIOMotor.Read());
			loaderHandler();
			m_pTimer->SimpleusDelay(200*1000);
			shiftreg=0;
			tapecartMode=MODE_STREAM;
			GPIOLEDMotor.Write(!GPIOMotor.Read());
			if(m_pTap->sense_on) GPIOSense.Write(LOW);
			else GPIOSense.Write(HIGH);
			break;
		case MODE_C64COMMAND:
			// ensure that the motor signal can be turned off
			m_pTimer->SimpleusDelay(10*1000);
			GPIOLEDMotor.Write(!GPIOMotor.Read());
			c64CommandHandler();
			shiftreg=0;
			tapecartMode=MODE_STREAM;
			GPIOLEDMotor.Write(!GPIOMotor.Read());
			if(m_pTap->sense_on) GPIOSense.Write(LOW);
			else GPIOSense.Write(HIGH);
			break;
		default:
			break;
	}
}

