//
// netutils.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

//#include <circle/genericlock.h>
#include <circle/net/netconfig.h>
#include <circle/time.h>

#include "netutils.h"

NetUtils::NetUtils (CNetSubSystem *pNetSubSystem, CTimer *pTimer)
	:
	m_pNet (pNetSubSystem),
	m_pTimer (pTimer)
{
}

NetUtils::~NetUtils (void)
{
}

boolean NetUtils::Initialize (void)
{
	dnsClient=new CDNSClient(m_pNet);
	ntpClient=new CNTPClient(m_pNet);
	return TRUE;
}

void NetUtils::getNetStatus (void)
{
	connected=m_pNet->IsRunning();
	hostname=m_pNet->GetHostname();
	CNetConfig *c=m_pNet->GetConfig();

	ipAddr=(u32)*(c->GetIPAddress());
	gateway=(u32)*(c->GetDefaultGateway());
	dnsServer=(u32)*(c->GetDNSServer());
	broadcastAddr=(u32)*(c->GetBroadcastAddress());

	netMask =(u8)*(c->GetNetMask());
	netMask|=(u8)*(c->GetNetMask()+1)<<8;
	netMask|=(u8)*(c->GetNetMask()+2)<<16;
	netMask|=(u8)*(c->GetNetMask()+3)<<24;
}

unsigned NetUtils::setNTPTime (void)
{
	lastNTPSync=0;

	CIPAddress ntpIP;
	if(!(dnsClient->Resolve("pool.ntp.org", &ntpIP))) return 0;

	unsigned time=ntpClient->GetTime(ntpIP);
	if(time) m_pTimer->SetTime(time, FALSE);

	lastNTPSync=time;
	return time;
}

void NetUtils::setTimeZone (int diff)
{
	unsigned t=m_pTimer->GetUniversalTime();
	m_pTimer->SetTimeZone(diff);
	m_pTimer->SetTime(t, FALSE);
}

u32 NetUtils::DNSResolve(const char *name)
{
	CIPAddress IP;
	if(!(dnsClient->Resolve(name, &IP))) return 0;
	return (u32)IP;
}

void NetUtils::runJob()
{
	jobLock.Acquire();
	switch(pendingJob)
	{
		case NET_DNS_RESOLVE:
			DNSResolveResult=DNSResolve(DNSResolveName);
			break;
		case NET_NTP_SYNC:
			setNTPTime();
			break;
		case NET_STATUS:
			getNetStatus();
			break;
		default:
			break;
	}
	pendingJob=NET_NONE;
	jobLock.Release();
}

void NetUtils::tcGetTime(u8 *timeData)
{
	CTime t;
	t.Set(m_pTimer->GetLocalTime());

  /* 
   *   2 year (YYYY)
   *   1 century
   *   1 year%100 (YY)
   *   1 month (1-12)
   *   1 day of month
   *   1 hours
   *   1 minutes
   *   1 seconds
   *   1 hours (bcd, bit 7 0=am, 1=pm)
   *   1 minutes (bcd)
   *   1 seconds (bcd)
   *   1 weekday (0=Sunday)
   *   4 seconds since 1970 (unsigned)
   *   1 sync status
   *     0-254 hours since last ntp sync
   *     255 not synced
   */
	unsigned year=t.GetYear();
	timeData[0]=year&0xff;
	timeData[1]=year>>8;

	unsigned ymod100=year%100;
	unsigned century=(year-ymod100)/100;
	timeData[2]=century;
	timeData[3]=ymod100;

	unsigned month=t.GetMonth();
	timeData[4]=month;

	unsigned monthDay=t.GetMonthDay();
	timeData[5]=monthDay;

	unsigned hours=t.GetHours();
	timeData[6]=hours;

	unsigned minutes=t.GetMinutes();
	timeData[7]=minutes;

	unsigned seconds=t.GetSeconds();
	timeData[8]=seconds;

	unsigned hours12=hours;
	boolean pm=FALSE;
	if(hours>=12)
	{
		pm=TRUE;
		hours12-=12;
	}
	if(hours12==0) hours12=12;
	unsigned hoursmod10=hours12%10;
	unsigned hoursBCD=(((hours12-hoursmod10)/10)<<4)|hoursmod10;
	if(pm) hoursBCD|=0x80;
	timeData[9]=hoursBCD;

	unsigned minutesmod10=minutes%10;
	unsigned minutesBCD=(((minutes-minutesmod10)/10)<<4)|minutesmod10;
	timeData[10]=minutesBCD;

	unsigned secondsmod10=seconds%10;
	unsigned secondsBCD=(((seconds-secondsmod10)/10)<<4)|secondsmod10;
	timeData[11]=secondsBCD;

	unsigned weekday=t.GetWeekDay();
	timeData[12]=weekday;

	unsigned unixtime=t.Get();
	timeData[13]=unixtime&0xff;
	timeData[14]=unixtime>>8;
	timeData[15]=unixtime>>16;
	timeData[16]=unixtime>>24;

	if(lastNTPSync==0) timeData[17]=255;
	else {
		unsigned syncAge=(m_pTimer->GetUniversalTime())-lastNTPSync;
		unsigned syncAgeHrs=(syncAge-(syncAge%(60*60)))/(60*60);
		if(syncAgeHrs>255) timeData[17]=255;
		else timeData[17]=syncAgeHrs;
	}
}

