//
// lcd.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

#include <circle/spinlock.h>
#include <string.h>

#include "lcd.h"

CSpinLock lcdSpinLock;

CGPIOPin GPIOButton[5];
int pinButtons[5];
boolean buttonState[5];
boolean buttonLastState[5];
u64 buttonLastDebounce[5];

u64 debounceDelay=50000;

u8 chargen[] = {
#include "c64/chargen.h"
};

LCD::LCD (CTimer *pTimer)
	:
		m_pTimer	(pTimer),
		m_i2c		(1, TRUE),
		m_lcd		(&m_i2c, lcdWidth, lcdHeight)
{
}

LCD::~LCD (void)
{
}

boolean LCD::Initialize (void)
{
	lcdConnected=m_lcd.Initialize();
	if(lcdConnected) m_lcd.Clear(0);

	this->setContent(0, "PiTap");
	this->setContent(1, "");
	this->setContent(2, "");
	this->setContent(3, "");
	this->setContent(4, "");
	this->setContent(5, "");
	this->setContent(6, "");
	this->setContent(7, "");

	pinButtons[0]=PIN_BUTTON1;
	pinButtons[1]=PIN_BUTTON2;
	pinButtons[2]=PIN_BUTTON3;
	pinButtons[3]=PIN_BUTTON4;
	pinButtons[4]=PIN_BUTTON5;

	for(int i=0; i<5; i++)
	{
		GPIOButton[i].AssignPin(pinButtons[i]);
		GPIOButton[i].SetMode(GPIOModeInputPullUp);
		buttonState[i]=FALSE;
		buttonLastState[i]=FALSE;
		buttonLastDebounce[i]=FALSE;
		buttonOn[i]=FALSE;
		buttonDepress[i]=FALSE;
		buttonPressTime[i]=0;
	}

	return TRUE;
}

void LCD::setContent (unsigned line, const char *buf, boolean reverse)
{
	if(!lcdConnected) return;
	if((line*fontHeight)>=lcdHeight) return;

	lcdSpinLock.Acquire();

	char buf2[255];
	char lineBuf[255];
	unsigned maxLen=lcdWidth/fontWidth;

	strncpy(buf2, buf, 255);
	if(strlen(buf2)>maxLen) strcpy(buf2+maxLen-4, buf+strlen(buf)-4);

	unsigned i;
	for(i=0; buf2[i] && i<maxLen; i++) lineBuf[i]=buf2[i];
	for(   ;            i<maxLen; i++) lineBuf[i]='\0';

	for(unsigned i=0; i<maxLen; i++)
	{
		unsigned c=lineBuf[i];
		if(c==0) c=32;
		else if(c>=97 && c<=122) c-=96;
		c+=256;
		if(reverse) c+=128;
		unsigned fontIndex=c*((fontWidth/8)*fontHeight);
		m_lcd.SetArea({(i*fontWidth), (i*fontWidth)+fontWidth-1, line*8, ((line+1)*8)-1}, (const void *)(chargen+fontIndex));
	}

	lcdSpinLock.Release();
}

void LCD::updateButtons (void)
{
	u64 timeNow=m_pTimer->GetClockTicks64();

	for(int i=0; i<5; i++)
	{
		boolean button=!GPIOButton[i].Read();
		if(button!=buttonLastState[i])
		{
			buttonLastDebounce[i]=timeNow;
		}
		buttonLastState[i]=button;
		if((timeNow - buttonLastDebounce[i]) > debounceDelay)
		{
			if(button!=buttonState[i])
			{
				buttonState[i]=button;
				if(buttonState[i])
				{
					buttonOn[i]=TRUE;
					buttonPressTime[i]=timeNow;
				} else {
					if((timeNow-buttonPressTime[i])<1500000) buttonDepress[i]=TRUE;
					buttonPressTime[i]=0;
				}
			}
		}
	}
}

