//
// kernel.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//


#include "kernel.h"
#include "webserver.h"
#include "display.h"

// Network configuration
#define USE_DHCP

#ifndef USE_DHCP
static const u8 IPAddress[]      = {192, 168, 1, 250};
static const u8 NetMask[]        = {255, 255, 255, 0};
static const u8 DefaultGateway[] = {192, 168, 1, 1};
static const u8 DNSServer[]      = {192, 168, 1, 1};
#endif

#define DRIVE		"SD:"
#define FIRMWARE_PATH	DRIVE "/firmware/"		// firmware files must be provided here
#define CONFIG_FILE	DRIVE "/wpa_supplicant.conf"

static const char FromKernel[] = "kernel";

CKernel::CKernel (void)
:       
        m_Timer (&m_Interrupt),
        //m_Logger (m_Options.GetLogLevel (), &m_Timer),
        m_Logger (0, &m_Timer),
	m_USBHCI (&m_Interrupt, &m_Timer),
	m_EMMC (&m_Interrupt, &m_Timer, &m_ActLED),
	m_WLAN (FIRMWARE_PATH),
#ifndef USE_DHCP
	m_Net (IPAddress, NetMask, DefaultGateway, DNSServer),
#endif
	m_Net (0, 0, 0, 0, "pitap", NetDeviceTypeWLAN),

	m_WPASupplicant (CONFIG_FILE),
	m_NetUtils (&m_Net, &m_Timer),
	m_Tap (&m_Timer),
	m_TapeCart (&m_Timer, &m_File, &m_Tap, &m_Status, &m_TapUtils, &m_NetUtils, &m_ActLED),
	m_TapCores (CMemorySystem::Get(), &m_Timer, &m_Logger, &m_Tap, &m_TapeCart, &m_Display),
	m_FS (&m_Interrupt, &m_Timer, &m_ActLED, &m_File, &m_Status, &m_Tap, &m_TapeCart, &m_TapUtils),
	m_LCD (&m_Timer),
	m_Display (&m_DeviceNameService, &m_File, &m_FS, &m_Tap, &m_TapeCart, &m_Status, &m_Timer, &m_LCD)
{
}

CKernel::~CKernel (void)
{
}

boolean CKernel::Initialize (void)
{
        boolean bOK = TRUE;

        if (bOK) bOK = m_Serial.Initialize (115200);

        if (bOK)
        {
                CDevice *pTarget = m_DeviceNameService.GetDevice (
                                        m_Options.GetLogDevice (), FALSE);
                bOK = m_Logger.Initialize (pTarget);
        }

        if (bOK) bOK = m_Interrupt.Initialize ();

        if (bOK)
        {
                bOK = m_Timer.Initialize ();
        }

	if (bOK)
	{
		bOK = m_USBHCI.Initialize ();
	}

	if (bOK)
	{
		bOK = m_EMMC.Initialize ();
	}
	// Mount file system
	if (f_mount (&m_FileSystem, DRIVE, 1) != FR_OK)
	{
		m_Logger.Write (FromKernel, LogPanic, "Cannot mount drive %s", DRIVE);
		bOK = FALSE;
	} else {
		m_Logger.Write (FromKernel, LogNotice, "Successfully mounted drive %s", DRIVE);
	}

	if (bOK) bOK = m_Status.Initialize();

	if (bOK) bOK = m_WLAN.Initialize ();
	if (bOK) bOK = m_Net.Initialize (FALSE);
	if (bOK) bOK = m_WPASupplicant.Initialize ();

	if (bOK) bOK = m_NetUtils.Initialize ();
	if (bOK) bOK = m_Tap.Initialize ();
	if (bOK) bOK = m_TapeCart.Initialize ();
	if (bOK) bOK = m_Display.Initialize ();
	if (bOK) bOK = m_FS.Initialize ();
	if (bOK) bOK = m_LCD.Initialize ();

        return bOK;
}

TShutdownMode CKernel::Run (void)
{
	m_TapCores.Initialize();

	boolean webServerStarted=FALSE;
	CString IPString;

	boolean isPlaying=FALSE;
	boolean isRecording=FALSE;
	boolean isMotorOn=FALSE;
	unsigned counterSeconds=0;
	tapecartMode_t tapecartMode=MODE_STREAM;
	boolean updateLCD=TRUE;
	CString LCDString;

	if(m_File.fileStat("/pitap/browser.prg"))
		m_FS.mountFile("/pitap/browser.prg");

	for (unsigned nCount = 0; 1; nCount++)
	{
		m_Scheduler.Yield ();

		m_Scheduler.MsSleep (100);

		if(nCount%10==0)
		{
			int i;
			i=m_CPUThrottle.GetTemperature();
			if(i>0) m_Status.cpuTemp=i;
			i=m_CPUThrottle.GetClockRate();
			if(i>0) m_Status.clockRate=i;
		}

		m_LCD.updateButtons();

		counterSeconds=m_Tap.getCounterSeconds();
		isPlaying=m_Tap.isPlaying();
		isRecording=m_Tap.isRecording();
		isMotorOn=m_Tap.getMotor();

		if(m_Status.tapecartFileChanged)
		{
			m_FS.mountTapecart(m_Status.currentTapecart);
			m_Status.tcrtLock.Acquire();
			m_Status.tapecartFileChanged=FALSE;
			m_Status.tcrtLock.Release();
		}

		if(m_NetUtils.pendingJob!=NET_NONE)
		{
			m_NetUtils.runJob();
		}

		if(m_Status.counterSeconds!=counterSeconds)
		{
			m_Status.counterSeconds=counterSeconds;
			updateLCD=TRUE;
		}
		if(m_Status.isPlaying!=isPlaying)
		{
			m_Status.isPlaying=isPlaying;
			updateLCD=TRUE;
		}
		if(m_Status.isRecording!=isRecording)
		{
			m_Status.isRecording=isRecording;
			updateLCD=TRUE;
		}
		if(m_Status.isMotorOn!=isMotorOn)
		{
			m_Status.isMotorOn=isMotorOn;
			updateLCD=TRUE;
		}
		if(m_TapeCart.tapecartMode!=tapecartMode)
		{
			tapecartMode=m_TapeCart.tapecartMode;
			updateLCD=TRUE;
		}
		if(updateLCD)
		{
			if(tapecartMode!=MODE_STREAM)
					     LCDString.Format("Tapecart     %03d", counterSeconds);
			else if(isPlaying)   LCDString.Format("Playing      %03d", counterSeconds);
			else if(isRecording) LCDString.Format("Recording    %03d", counterSeconds);
			else                 LCDString.Format("Stopped      %03d", counterSeconds);
			m_LCD.setContent(7, LCDString, isMotorOn);
			updateLCD=FALSE;
		}

		if(m_Status.fileChanged)
		{
			m_LCD.setContent(6, m_Status.currentTapShort, FALSE);
			m_Status.fileChanged=FALSE;
		}

		if(!webServerStarted) {
			if(m_Net.IsRunning())
			{
				m_Net.GetConfig ()->GetIPAddress ()->Format (&IPString);
				new CWebServer (&m_Net, &m_Timer, &m_ActLED, &m_File, &m_FS, &m_Tap, &m_TapeCart, &m_Status);
				m_Status.ipAddress=IPString;
				m_LCD.setContent(0, m_Status.ipAddress, FALSE);
				webServerStarted=TRUE;
				m_NetUtils.setNTPTime();
			}
		}

		if(m_Status.rebootRequest)
		{
			m_Logger.Write (FromKernel, LogNotice, "Rebooting");
			m_Scheduler.MsSleep (2000);
			return ShutdownReboot;
		}
	}

        return ShutdownHalt;
}

