//
// tapecart.h
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

// Adapted from Tapecart by Ingo Korb
// https://github.com/ikorb/tapecart
// and Tapecart SD by Kim Jorgensen
//  https://github.com/KimJorgensen/tapecart

#ifndef _tapecart_h
#define _tapecart_h

#include <circle/types.h>

#include "file.h"
#include "status.h"
#include "tap.h"
#include "taputils.h"

#define TCRT_HDR	  0
#define TCRT_VER	 16
#define TCRT_OFFSET	 18
#define TCRT_DATALEN	 20
#define TCRT_CALLADDR	 22
#define TCRT_NAME	 24
#define TCRT_FLAGS	 40
#define TCRT_LOADER	 41
#define TCRT_FLASHLEN	212
#define TCRT_DAT	216

#define DEBUGFLAG_SEND_CMDOK    (1 << 0)
#define DEBUGFLAG_BLINK_MAGIC   (1 << 1)
#define DEBUGFLAG_BLINK_COMMAND (1 << 2)

typedef enum {
  MODE_STREAM,
  MODE_LOADER,
  MODE_C64COMMAND
} tapecartMode_t;

typedef enum {
  CMD_EXIT = 0,
  CMD_READ_DEVICEINFO,
  CMD_READ_DEVICESIZES,
  CMD_READ_CAPABILITIES,

  CMD_READ_FLASH  = 0x10,
  CMD_READ_FLASH_FAST,
  CMD_WRITE_FLASH,
  CMD_WRITE_FLASH_FAST, // Not Yet Implemented
  CMD_ERASE_FLASH_64K,
  CMD_ERASE_FLASH_BLOCK,
  CMD_CRC32_FLASH,

  CMD_READ_LOADER = 0x20,
  CMD_READ_LOADINFO,
  CMD_WRITE_LOADER,
  CMD_WRITE_LOADINFO,

  CMD_LED_OFF = 0x30,
  CMD_LED_ON,
  CMD_READ_DEBUGFLAGS,
  CMD_WRITE_DEBUGFLAGS,

  CMD_DIR_SETPARAMS = 0x40,
  CMD_DIR_LOOKUP,
  
  /* SD commands */
  CMD_SD_OPEN_DIR = 0x80,
  CMD_SD_READ_DIR_FAST,
  CMD_SD_SELECT_FILE,

  /* internal use only */
  CMD_RAMEXEC = 0xf0,
} command_t;

typedef enum {
  FILE_NONE = 0x00,
  FILE_DIR,
  FILE_PRG,
  FILE_TCRT,
  FILE_SID,
  FILE_UNKNOWN = 0xFF
} file_t;


class TapeCart
{
public:
	TapeCart (CTimer *pTimer, File *m_pFile, Tap *pTap, Status *pStatus, TapUtils *pTapUtils, CActLED *pActLED);
	~TapeCart (void);

	boolean Initialize (void);

	void hwInterface (void);
	unsigned mountTapecart (u8 *tapecart, const unsigned size);
	//void setMode(tapecartMode_t mode);

	u16 shiftreg=0;
	//u32 tcrtStatus=0;
	//u32 tcrtCount=0;
	CString tcrtLog;

	tapecartMode_t tapecartMode=MODE_STREAM;

	CString	currentDir="/";

private:
	s16 c64_get_byte_error(void);
	s16 c64_get_byte(void);
	boolean get_u8(u8 *result);
	boolean get_u16(u16 *result);
	boolean get_u24(u32 *result);

	void tx_byte_common(u8 b);
	boolean tx_byte_loader(u8 b);
	boolean tx_byte_command(u8 b);
	boolean c64_send_byte(u8 v);
	boolean c64_send_u16(u16 v);
	boolean c64_send_u24(u32 v);
	boolean c64_send_u32(u32 v);

	void checkShiftreg(void);
	void c64CommandHandler(void);
	void commandHandler(void);
	void loaderHandler(void);
	void sendPulse(void);

	u32 crc32b(u8 *m, unsigned len);
	void crc32_flash(void);
	void do_dir_lookup(void);
	void erase_flash_64k(void);
	void erase_flash_block(void);
	void read_capabilities(void);
	void read_debugflags(void);
	void read_deviceinfo(void);
	void read_devicesizes(void);
	void read_flash(boolean fast);
	void read_loader(void);
	void read_loadinfo(void);
	void sd_open_dir(void);
	void sd_read_dir(void);
	void sd_select_file(void);
	void set_dirparams(void);
	void setLED(boolean v);
	void write_debugflags(void);
	void write_flash(void);
	void write_loader(void);
	void write_loadinfo(void);

	void writeTapecartFile(void);

	CGPIOPin GPIOMotor;
	CGPIOPin GPIORead;
	CGPIOPin GPIOWrite;
	CGPIOPin GPIOSense;

	CGPIOPin GPIOLEDMotor;
	CGPIOPin GPIOLEDRecord;
	CGPIOPin GPIOLEDPlay;

	u16 maxFiles=1000;

	u8 *tcrtData=0;
	unsigned tcrtSize=0;

	u16 debug_flags=0;
	u32 dir_base=0;
	u16 dir_entries=0;
	u8 dir_name_len=0;
	u8 dir_data_len=0;

	CTimer	*m_pTimer;
	File	*m_pFile;
	Tap	*m_pTap;
	Status	*m_pStatus;
	TapUtils *m_pTapUtils;
	CActLED	*m_pActLED;
};

#endif

